function h=m_plot_track(lat,lon,options)
% Purpose: plot ship track on map
%
% Variables:
%	- lat: vector of latitudes
%	- lon: vector of longitudes
%	- options: plotting options: 'limits' (optional)
%	- h: figure handler
%
% Global variables: DATA_DIR, MAP_LAT_PADDING, MAP_LON_PADDING

% Global variables
global DATA_DIR
global MAP_LAT_PADDING
global MAP_LON_PADDING

% Processing argument
if nargin <2 | nargin>3
   error('Bad number of arguments\n');
end

h=figure;
set(gcf,'PaperType','a4');
hold on;

% Axes
lat_south=min(lat)-MAP_LAT_PADDING;
lat_north=max(lat)+MAP_LAT_PADDING;
long_west=min(lon)-MAP_LON_PADDING;
long_east=max(lon)+MAP_LON_PADDING;

% Plotting
m_proj('mercator','longitude',[long_west long_east],'latitude',[lat_south lat_north]);

% 2min topo_8.2 bathymetry
[latgrat,longrat,map]=read_topo82([lat_south lat_north],[long_west long_east]);
[c,h1]=m_contour(longrat,latgrat,map,[-5000:500:-1000 -800:200:-200 -150:50:-50],'k');
clabel(c,h1,'Color','k','FontSize',6)

% Coast lines
m_usercoast([DATA_DIR filesep 'gshhs_h_medit.mat'],'patch',[.7 .7 .7],'edgecolor','k');
m_grid('box','fancy','tickdir','out');

% Track
m_plot(lon,lat,'r');

% Optional limits
if (nargin==3 & options=='limits')
   m_text(lon(1),lat(1),'Dbut \rightarrow','Color','b','FontSize',8,'HorizontalAlignment','right'); 
   m_text(lon(length(lon)),lat(length(lat)),'\leftarrow Fin','Color','b','FontSize',8); 
end

% Legends
title('Route du TETHYS II');
zoom on;

% Saving data in figure
s.coord=[lat lon];
set(h,'UserData',s);

%%%%%%%%%%%%%%%%%%%%
% Button bar layout
%%%%%%%%%%%%%%%%%%%%

buttons(1).str='Position';
buttons(1).callback='call_positions';

buttons(2).str='Distance';
buttons(2).callback='call_distance';

buttons(3).str='Ensemble';
buttons(3).callback='call_ensemble';

h_button_bar(gcf,buttons);
